package com.creamtec.ajaxswing.support.components;

import java.awt.BorderLayout;
import java.awt.Component;
import java.util.HashMap;
import java.util.Map;

import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.JTree;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

/**
 * Custom component that mimics expanding Outlook folders
 *
 */
public class AccordionPane extends JPanel implements TreeSelectionListener {
    JTabbedPane tabbedPane = new JTabbedPane();
    Map menuItems = new HashMap();
    
    // Use HTML to match the height with CSS and ensure that each tab is on it's on line (for sizing of children)
    public static final String TITLE_PREFIX = "<html><span style=\"font-size:14px\">";
    public static final String TITLE_SUFFIX = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;</span></html>";
    
    public AccordionPane() {
        setLayout(new BorderLayout());
        add(tabbedPane, BorderLayout.CENTER);
    }
    
    public AccordionPane(JMenuBar menuBar) {
        this();
        addMenus(menuBar);
    }
    
    /**
     * Adds a new tab to the pane 
     * @param title tab title
     * @param component tab contents
     */
    public void add(String title, JComponent component) {
        // Add extra spaces to ensure that tabs wrap lines
        this.tabbedPane.add(createTabTitle(title), component);
    }
    
    /**
     * Returns the content of the tab
     * @param i tab index
     * @return
     */
    public Component getComponentAt(int i) {
        return this.tabbedPane.getComponentAt(i);
    }
    
    /**
     * Sets the component for the specified tab 
     * @param index tab index
     * @param component
     */
    public void setComponentAt(int i, Component component) {
        this.tabbedPane.setComponentAt(i, component);
    }
    
    /**
     * Returns the number of tabs
     * @return
     */
    public int getTabCount() {
        return this.tabbedPane.getTabCount();
    }
    
    /**
     * Returns tab title
     * @param i tab index
     * @return
     */
    public String getTitleAt(int i) {
        String title = this.tabbedPane.getTitleAt(i);
        return title.substring(TITLE_PREFIX.length(), title.length() - TITLE_SUFFIX.length());
    }
    
    public int getSelectedIndex() {
        return this.tabbedPane.getSelectedIndex();
    }
    
    public void setSelectedIndex(int index) {
        this.tabbedPane.setSelectedIndex(index);
    }
    
    public void removeTabAt(int index) {
        this.tabbedPane.removeTabAt(index);
    }
    
    public void insertTab(String title, Icon icon, Component component, String tip, int index) {
        this.tabbedPane.insertTab(createTabTitle(title), icon, component, tip, index);
    }
    
    /**
     * Adds the tab with the menu title and all of it's children represented as a tree
     * @param menu menu to import
     * @return
     */
    public JMenu add(JMenu menu) {
        Tree tree = createTreeForMenu(menu);
        add(menu.getText(), tree);
        return menu;
    }

    private Tree createTreeForMenu(JMenu menu) {
        TreeNode root = new TreeNode();
        addJMenuItems(menu, root);
        Tree tree = new Tree(root);
        tree.setRootVisible(false);
        tree.addTreeSelectionListener(this);
        return tree;
    }
    
    public void update(JMenu menu) {
        int index = this.tabbedPane.indexOfTab(createTabTitle(menu.getText()));
        if (index != -1) {
            Component tab = this.tabbedPane.getComponentAt(index);
            if (tab instanceof Tree) {
                Tree tree = (Tree)tab;
                TreeNode root = (TreeNode) tree.getModel().getRoot();
                root.removeAllChildren();
                addJMenuItems(menu, root);
                DefaultTreeModel model = (DefaultTreeModel) tree.getModel();
                model.reload();
            }
        }
    }

    private String createTabTitle(String title) {
        return TITLE_PREFIX + title + TITLE_SUFFIX;    
    }

    private void addJMenuItems(JMenu menu, TreeNode root) {
        for (int i = 0; i < menu.getItemCount(); i++) {
            JMenuItem item = menu.getItem(i);
            if (item != null) {
                TreeNode itemNode = new TreeNode(item.getText(), item);
                root.add(itemNode);
                // Store for fast lookup on update
                this.menuItems.put(Integer.toString(item.hashCode()), item);
                
                if (item instanceof JMenu)
                    addJMenuItems((JMenu) item, itemNode);
            }
        }
    }
    
    /**
     * Adds all menus of the menu bar as tabs with trees of items
     * @param menuBar
     */
    public void addMenus(JMenuBar menuBar) {
        for(int i = 0; i < menuBar.getMenuCount(); i++) {
            add(menuBar.getMenu(i));
        }
    }
    
    /**
     * Simulates menu item being clicked
     * @param menuItemHashCode hash code of the menu item to click
     */
    public void doClick(String menuItemHashCode) {
        JMenuItem item  = (JMenuItem) this.menuItems.get(menuItemHashCode);
        item.doClick();
    }
    
    /**
     * Marker interface for accordion tree
     * @author Alex
     *
     */
    public static class Tree extends JTree {
        public Tree(TreeNode root) {
            super(root);
        }
    }
    
    /**
     * Marker interface for accordion tree node
     * @author Alex
     *
     */
    public static class TreeNode extends DefaultMutableTreeNode {
        protected String text;
        
        public TreeNode() {
        }
        
        public TreeNode(String text, Object userObject) {
            super(userObject);
            this.text = text;
        }
        
        public String toString() {
            return this.text;
        }
    }

    public void valueChanged(TreeSelectionEvent e) {
        if (e.getNewLeadSelectionPath() != null) {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode) e.getNewLeadSelectionPath().getLastPathComponent();
            JMenuItem item = (JMenuItem) node.getUserObject();
            item.doClick();
        }
        
    }
    
}